import asyncio

from pylog.pylogger import PyLogger

from urpc_enum.corexymoverparameter import CoreXYMoverParameter
from urpc_enum.dualmoverparameter import DualMoverParameter
from urpc_enum.moverparameter import MoverParameter

from bodhi.common.firmware_util import *


# From the terminal use the names defined in the endpoint dictionaries to access the endpoints directly.
def __getattr__(name):
    if name in node_endpoints:
        return get_node_endpoint(name)
    if name in mover_endpoints:
        return get_mover_endpoint(name)
    if name in dual_mover_endpoints:
        return get_dual_mover_endpoint(name)
    if name in measurement_endpoints:
        return get_measurement_endpoint(name)
    if name in serial_endpoints:
        return get_serial_endpoint(name)
    if name in system_control_endpoints:
        return get_system_control_endpoint(name)
    if name in fan_control_endpoints:
        return get_fan_control_endpoint(name)
    if name in temperature_control_endpoints:
        return get_temperature_control_endpoint(name)
    raise AttributeError(f"module '{__name__}' has no attribute '{name}'")


# Initialization #######################################################################################################

async def init(*endpoints):
    for endpoint in endpoints:
        if endpoint in node_endpoints:
            PyLogger.logger.info(f"Start '{endpoint}' firmware")
            await start_firmware(endpoint)
        if endpoint in mover_endpoints:
            PyLogger.logger.info(f"Initialize '{endpoint}' Mover")
            mover = get_mover_endpoint(endpoint)
            if endpoint == 'excmc':
                await mover.SetProfile(
                    # speed = 30000 usteps/s / 256 usteps/steps * 0.225 °/steps = 26.37 °/s
                    # power = 73% -> 670 mA
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=73, holdPower=25, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,              450000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,       450000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'emsmc':
                await mover.SetProfile(
                    # speed = 30000 usteps/s / 256 usteps/steps * 0.225 °/steps = 26.37 °/s
                    # power = 73% -> 670 mA
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=73, holdPower=25, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,              450000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,       450000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'excls':
                await mover.SetProfile(
                    # speed = 30000 usteps/s / 256 usteps/steps * 0.225 °/steps = 26.37 °/s
                    # power = 88% -> 800 mA
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=88, holdPower=30, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,               55000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,        55000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'emsls':
                await mover.SetProfile(
                    # speed = 30000 usteps/s / 256 usteps/steps * 0.225 °/steps = 26.37 °/s
                    # power = 88% -> 800 mA
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=88, holdPower=30, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,               55000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,        55000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'xflfw':
                await mover.SetProfile(
                    # speed = 10000 usteps/s / 256 usteps/steps * 1.8 °/steps = 70.31 °/s
                    # power = 88% -> 800 mA
                    handle=0, speed=10000, accel=100000, decel=100000, uSteps=256, drivePower=88, holdPower=30, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,              110000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,       110000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x82)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'pmtfw':
                await mover.SetProfile(
                    # speed = 30000 usteps/s / 256 usteps/steps * 0.75 °/steps = 87.89 °/s
                    # power = 66% -> 600 mA
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=66, holdPower=22, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,              135000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,       135000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 1)
                await mover.SetConfigurationStatus(1)
            else:
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(MoverParameter.HomeSearchDirection,               0)
                await mover.SetParameter(MoverParameter.HomeMaxDistance,             1000000)
                await mover.SetParameter(MoverParameter.HomeMaxReverseDistance,      1000000)
                await mover.SetParameter(MoverParameter.HomeExtraReverseDistance,          0)
                await mover.SetParameter(MoverParameter.HomeCalibrationSpeed,          10000)
                await mover.SetParameter(MoverParameter.HomeSensorEnable,               0x01)
                await mover.SetParameter(MoverParameter.MovementDirection,                 0)
                await mover.SetConfigurationStatus(1)
        if endpoint in dual_mover_endpoints:
            PyLogger.logger.info(f"Initialize '{endpoint}' Dual Mover")
            mover = get_dual_mover_endpoint(endpoint)
            if endpoint == 'umh':
                await mover.SetProfile(
                    # speed = 60000 usteps/s / 256 usteps/steps * 0.0127 mm/steps = 2.98 mm/s
                    # speed = 60000 usteps/s / 256 usteps/steps * 0.0063 mm/steps = 1.48 mm/s (Demo 1)
                    # power = 125% -> 420 mA
                    handle=0, speed=60000, accel=300000, decel=300000, uSteps=256, drivePower=125, holdPower=42, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(DualMoverParameter.LinkModeA,                        0)
                await mover.SetParameter(DualMoverParameter.LinkModeB,                        1)
                await mover.SetParameter(DualMoverParameter.RampModeA,                        1)
                await mover.SetParameter(DualMoverParameter.RampModeB,                        1)
                await mover.SetParameter(DualMoverParameter.HomeMoverOrder,                   2)   # Only home X-Axis !!!
                await mover.SetParameter(DualMoverParameter.HomeSearchDirectionA,             0)
                await mover.SetParameter(DualMoverParameter.HomeMaxDistanceA,           4500000)
                await mover.SetParameter(DualMoverParameter.HomeMaxReverseDistanceA,    4500000)
                await mover.SetParameter(DualMoverParameter.HomeCalibrationSpeedA,        10000)
                await mover.SetParameter(DualMoverParameter.HomeSensorEnableA,             0x01)
                await mover.SetParameter(DualMoverParameter.MovementDirectionA,               0)
                await mover.SetParameter(DualMoverParameter.MovementDirectionB,               0)
                await mover.SetConfigurationStatus(1)
            elif endpoint == 'st':
                await mover.SetProfile(
                    handle=0, speed=50000, accel=1500000, decel=1500000, uSteps=256, drivePower=66, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(DualMoverParameter.LinkModeA,                        2)
                await mover.SetParameter(DualMoverParameter.LinkModeB,                        1)
                await mover.SetParameter(DualMoverParameter.RampModeA,                        1)
                await mover.SetParameter(DualMoverParameter.RampModeB,                        1)
                await mover.SetParameter(DualMoverParameter.HomeMoverOrder,                   1)
                await mover.SetParameter(DualMoverParameter.HomeSearchDirectionA,             0)
                await mover.SetParameter(DualMoverParameter.HomeSearchDirectionB,             0)
                await mover.SetParameter(DualMoverParameter.HomeMaxDistanceA,            550000)
                await mover.SetParameter(DualMoverParameter.HomeMaxDistanceB,            550000)
                await mover.SetParameter(DualMoverParameter.HomeMaxReverseDistanceA,     550000)
                await mover.SetParameter(DualMoverParameter.HomeMaxReverseDistanceB,     550000)
                await mover.SetParameter(DualMoverParameter.HomeCalibrationSpeedA,        10000)
                await mover.SetParameter(DualMoverParameter.HomeCalibrationSpeedB,        10000)
                await mover.SetParameter(DualMoverParameter.HomeSensorEnableA,             0x01)
                await mover.SetParameter(DualMoverParameter.HomeSensorEnableB,             0x01)
                await mover.SetParameter(DualMoverParameter.EncoderModeA,                     1)
                await mover.SetParameter(DualMoverParameter.EncoderModeB,                     1)
                await mover.SetParameter(DualMoverParameter.EncoderCorrectionFactorA,      12.8)
                await mover.SetParameter(DualMoverParameter.EncoderCorrectionFactorB,      12.8)
                await mover.SetParameter(DualMoverParameter.MaxEncoderDeviationA,           512)
                await mover.SetParameter(DualMoverParameter.MaxEncoderDeviationB,           512)
                await mover.SetParameter(DualMoverParameter.MovementDirectionA,               0)
                await mover.SetParameter(DualMoverParameter.MovementDirectionB,               0)
                await mover.SetParameter(DualMoverParameter.RotationalDirectionA,             0)
                await mover.SetParameter(DualMoverParameter.RotationalDirectionB,             0)
                await mover.SetConfigurationStatus(1)
            else:
                await mover.SetProfile(
                    handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
                )
                await mover.UseProfile(0)
                await mover.SetParameter(DualMoverParameter.HomeMaxDistanceA,           1000000)
                await mover.SetParameter(DualMoverParameter.HomeMaxDistanceB,           1000000)
                await mover.SetParameter(DualMoverParameter.HomeCalibrationSpeedA,        10000)
                await mover.SetParameter(DualMoverParameter.HomeCalibrationSpeedB,        10000)
                await mover.SetParameter(DualMoverParameter.HomeSensorEnableA,             0x01)
                await mover.SetParameter(DualMoverParameter.HomeSensorEnableB,             0x01)
                await mover.SetConfigurationStatus(1)

    return f"init done"


# UMH Tests ############################################################################################################

async def umh_tilt_measurement():

    await init('eef', 'umh')

    umh = get_dual_mover_endpoint('umh')
    await umh.SetParameter(DualMoverParameter.HomeSensorEnableA, 0x02)
    await umh.Home()
    await umh.SetParameter(DualMoverParameter.HomeSensorEnableA, 0x01)
    tilt_usteps = (await umh.Home())[0]

    tilt_mm = tilt_usteps / 256 * 0.0127
    
    return f"umh_tilt = {tilt_usteps:.0f} µSteps -> {tilt_mm:.3f} mm"


async def umh_tilt_correction(tilt_usteps):

    await init('eef', 'umh')

    umh = get_dual_mover_endpoint('umh')
    a, b = await umh.GetPosition()              # tilt correction relative to curernt position
    await umh.Move(a, (b - tilt_usteps))        # correct the tilt using mover b
    await umh.SetPosition(a, 0)                 # set position b to 0, so subsequent Move(a,b) calls always use position b = 0
    await umh.Home()                            # return to home position
    
    return f"umh_tilt_correction done"


# Dual Mover Tests #####################################################################################################

async def dual_mover_test(mover_name='st', link_a=0, link_b=0, encoder_a=0, encoder_b=0):
    eef = get_node_endpoint('eef')
    mover = get_dual_mover_endpoint(mover_name)

    await eef.Reset()
    await asyncio.sleep(0.1)
    await start_firmware('eef')

    if mover_name == 'st':
        await mover.SetProfile(
            handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=40, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
        )
    if mover_name == 'umh':
        await mover.SetProfile(
            handle=0, speed=30000, accel=300000, decel=300000, uSteps=256, drivePower=120, holdPower=20, drivePowerHoldTime=1000, drivePowerFallTime=1000
        )
    await mover.UseProfile(0)
    await mover.SetParameter(DualMoverParameter.LinkModeA, link_a)
    await mover.SetParameter(DualMoverParameter.LinkModeB, link_b)
    await mover.SetParameter(DualMoverParameter.EncoderModeA, int(encoder_a != 0))
    await mover.SetParameter(DualMoverParameter.EncoderModeB, int(encoder_b != 0))
    await mover.SetParameter(DualMoverParameter.EncoderCorrectionFactorA, encoder_a)
    await mover.SetParameter(DualMoverParameter.EncoderCorrectionFactorB, encoder_b)
    await mover.SetParameter(DualMoverParameter.MaxEncoderDeviationA, 0xFFFFFFFF)
    await mover.SetParameter(DualMoverParameter.MaxEncoderDeviationB, 0xFFFFFFFF)
    await mover.SetConfigurationStatus(1)

    await mover.SetPosition(0, 0)
    PyLogger.logger.info(f"Mover: {await mover.GetPosition()}, Encoder: {await mover.GetEncoderPosition()}")
    
    await mover.Move(50000, 0)
    PyLogger.logger.info(f"Mover: {await mover.GetPosition()}, Encoder: {await mover.GetEncoderPosition()}")
    
    await mover.Move(50000, 50000)
    PyLogger.logger.info(f"Mover: {await mover.GetPosition()}, Encoder: {await mover.GetEncoderPosition()}")
    
    await mover.Move(0, 50000)
    PyLogger.logger.info(f"Mover: {await mover.GetPosition()}, Encoder: {await mover.GetEncoderPosition()}")
    
    await mover.Move(0, 0)
    PyLogger.logger.info(f"Mover: {await mover.GetPosition()}, Encoder: {await mover.GetEncoderPosition()}")

    return f"dual_mover_test done"


########################################################################################################################
